# Backend Directory Structure

```
c:\xampp\htdocs\Backend_school_portal\
├── config\
│   └── Database.php                    [NEW] MySQL connection class
├── api\
│   ├── student.php                     [NEW] Get student & parent info
│   ├── attendance.php                  [NEW] Get attendance summary
│   ├── results.php                     [NEW] Get academic results
│   ├── fees.php                        [NEW] Get fees information
│   └── upload_photo.php                [EXISTING] Photo upload endpoint
└── upload\
    └── photo\                          [Storage for uploaded photos]
```

## File Usage Guide

### config/Database.php
**Purpose**: Establishes connection to MySQL database
**Usage**: Required by all API endpoints
**Config**:
```php
private $host = 'localhost';           // XAMPP host
private $db_name = 'school_parent_portal';
private $user = 'root';                // XAMPP default user
private $password = '';                // XAMPP default password (empty)
```

### api/student.php
**Purpose**: Fetch student details and parent information
**Request**: GET /api/student.php?student_id=1
**Returns**: Student data + list of parents/guardians
**Database Tables**: students, parent_student, users, classes

### api/attendance.php
**Purpose**: Get attendance summary (present, absent, late, excused)
**Request**: GET /api/attendance.php?student_id=1
**Returns**: Count of attendance records by status
**Database Table**: attendance

### api/results.php
**Purpose**: Get academic results/grades
**Request**: GET /api/results.php?student_id=1&session_year=2025/2026&term=1st
**Returns**: Subjects with scores (CA1, CA2, Exam, Total, Grade)
**Database Table**: results

### api/fees.php
**Purpose**: Get school fees information
**Request**: GET /api/fees.php?student_id=1
**Returns**: All fees + payment status + summary totals
**Database Table**: fees

## HTTP Headers (Auto-configured)

All PHP files include CORS headers:
```php
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: GET, POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");
```

This allows Flutter app to make requests from any origin (localhost during dev).

## Error Responses

All endpoints follow this error format:
```json
{
  "success": false,
  "message": "Description of error"
}
```

Common errors:
- `Student ID is required` - No student_id parameter provided
- `Student not found` - Invalid student_id
- `Database Connection Error` - MySQL connection failed
- `Method not allowed` - Wrong HTTP method used

## Testing Commands

### Using Browser
```
http://localhost/Backend_school_portal/api/student.php?student_id=1
http://localhost/Backend_school_portal/api/attendance.php?student_id=1
http://localhost/Backend_school_portal/api/results.php?student_id=1
http://localhost/Backend_school_portal/api/fees.php?student_id=1
```

### Using cURL (Windows PowerShell)
```powershell
$uri = "http://localhost/Backend_school_portal/api/student.php?student_id=1"
Invoke-WebRequest -Uri $uri | ConvertFrom-Json | ConvertTo-Json
```

### Using Postman
1. Create new GET request
2. Set URL: http://localhost/Backend_school_portal/api/student.php?student_id=1
3. Send
4. View response in Body tab

## Database Schema Reference

### Key Tables for APIs

**students** table columns:
- id, admission_number, full_name, class_id, section, gender, date_of_birth, photo_url, blood_group, allergies, emergency_contact_name, emergency_contact_phone

**attendance** table columns:
- id, student_id, attendance_date, status (present/absent/late/excused), reason, marked_by

**results** table columns:
- id, student_id, session_year, term, subject, ca1, ca2, exam, total (auto), grade, position_in_class, teacher_remark, principal_remark

**fees** table columns:
- id, student_id, session_year, term, fee_type, amount_due, amount_paid, balance (auto), due_date, payment_date, payment_method, receipt_number

**parent_student** table (relationship):
- parent_id, student_id, relationship, is_primary

**users** table (for parents):
- id, email, phone, password_hash, full_name, role, otp_code, otp_expires_at, last_login, is_active

**classes** table:
- id, class_name, section, class_teacher_id

## Notes for cPanel Deployment

When moving to cPanel:

1. Upload files to: `public_html/Backend_school_portal/`
2. Create database on cPanel MySQL (using phpmyadmin)
3. Update config/Database.php with cPanel credentials:
   ```php
   private $host = 'localhost';  // Usually still localhost
   private $db_name = 'cpanel_username_school_portal';  // Format varies
   private $user = 'cpanel_username_school';
   private $password = 'your_db_password';
   ```
4. Update Flutter app baseUrl:
   ```dart
   static const String baseUrl = 'https://yourdomain.com/Backend_school_portal/api';
   ```

## Security Notes (Important for Production)

- ⚠️ Currently allows all CORS origins - restrict in production
- ⚠️ No authentication on endpoints - add JWT/session validation
- ⚠️ No SQL injection prevention - prepared statements already used ✓
- ⚠️ No rate limiting - consider adding
- ⚠️ Ensure HTTPS on production server
