<?php
/**
 * Login API Endpoint
 * Authenticates user against database
 */

header('Content-Type: application/json');
header("Access-Control-Allow-Origin: *");
header("Access-Control-Allow-Methods: POST, OPTIONS");
header("Access-Control-Allow-Headers: Content-Type");

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(200);
    exit();
}

require_once '../config/Database.php';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $db = new Database();
    $pdo = $db->connect();

    $input = json_decode(file_get_contents('php://input'), true);

    // DEBUG: Log incoming request
    error_log('🔐 LOGIN API: Incoming request');
    error_log('   Input: ' . json_encode($input));

    if (!isset($input['email']) || !isset($input['password'])) {
        error_log('❌ LOGIN API: Missing email or password');
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Email and password are required'
        ]);
        exit;
    }

    $email = filter_var($input['email'], FILTER_SANITIZE_EMAIL);
    $password = $input['password'];

    error_log('   Sanitized Email: ' . $email);
    error_log('   Password (first 3 chars): ' . substr($password, 0, 3) . '***');

    try {
        // Get user by email
        $query = "SELECT id, email, full_name, password_hash, role, is_active 
                  FROM users 
                  WHERE email = :email LIMIT 1";
        
        error_log('   Executing query to find user by email');
        
        $stmt = $pdo->prepare($query);
        $stmt->bindParam(':email', $email);
        $stmt->execute();
        
        $user = $stmt->fetch(PDO::FETCH_ASSOC);

        if (!$user) {
            error_log('❌ User not found: ' . $email);
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid email or password'
            ]);
            exit;
        }

        error_log('✅ User found: ' . $user['full_name']);
        error_log('   User ID: ' . $user['id']);
        error_log('   Role: ' . $user['role']);

        if (!$user['is_active']) {
            error_log('❌ Account inactive for user: ' . $email);
            http_response_code(403);
            echo json_encode([
                'success' => false,
                'message' => 'Account is inactive'
            ]);
            exit;
        }

        // Verify password
        error_log('   Verifying password...');
        if (!password_verify($password, $user['password_hash'])) {
            error_log('❌ Password verification failed for user: ' . $email);
            http_response_code(401);
            echo json_encode([
                'success' => false,
                'message' => 'Invalid email or password'
            ]);
            exit;
        }

        error_log('✅ Password verified successfully');

        // Get student(s) if parent
        $students = [];
        if ($user['role'] == 'parent' || $user['role'] == 'guardian') {
            error_log('   Fetching linked students...');
            $student_query = "SELECT s.id, s.admission_number, s.full_name, s.class_id, c.class_name, c.section
                             FROM parent_student ps
                             JOIN students s ON ps.student_id = s.id
                             LEFT JOIN classes c ON s.class_id = c.id
                             WHERE ps.parent_id = :parent_id";
            
            $student_stmt = $pdo->prepare($student_query);
            $student_stmt->bindParam(':parent_id', $user['id']);
            $student_stmt->execute();
            $students = $student_stmt->fetchAll(PDO::FETCH_ASSOC);
            error_log('   Found ' . count($students) . ' student(s)');
        }

        // Update last_login
        error_log('   Updating last_login timestamp...');
        $update_query = "UPDATE users SET last_login = NOW() WHERE id = :id";
        $update_stmt = $pdo->prepare($update_query);
        $update_stmt->bindParam(':id', $user['id']);
        $update_stmt->execute();

        $response = [
            'success' => true,
            'data' => [
                'user_id' => $user['id'],
                'id' => $user['id'],
                'email' => $user['email'],
                'full_name' => $user['full_name'],
                'role' => $user['role'],
                'students' => $students
            ]
        ];

        error_log('✅ LOGIN SUCCESSFUL');
        error_log('   Response: ' . json_encode($response));
        echo json_encode($response);

    } catch (Exception $e) {
        error_log('❌ Exception in login.php: ' . $e->getMessage());
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error: ' . $e->getMessage()
        ]);
    }
} else {
    http_response_code(405);
    echo json_encode([
        'success' => false,
        'message' => 'Method not allowed'
    ]);
}
?>
